<?php
/**
 * Admin Dashboard
 * Aegean Dominance Admin Panel
 */

session_start();
require_once __DIR__ . '/../config/config.php';

$auth = new Auth();
$auth->requireLogin();

$db = Database::getInstance();
$admin = $auth->getCurrentAdmin();

// Dashboard istatistikleri
try {
    $stats = [
        'services' => $db->fetchColumn("SELECT COUNT(*) FROM services WHERE is_active = 1"),
        'leads' => $db->fetchColumn("SELECT COUNT(*) FROM contact_forms WHERE status = 'new'"),
        'visitors_today' => $db->fetchColumn("SELECT COUNT(*) FROM visitors WHERE DATE(visit_date) = CURDATE()"),
        'visitors_month' => $db->fetchColumn("SELECT COUNT(*) FROM visitors WHERE MONTH(visit_date) = MONTH(CURDATE())"),
        'testimonials' => $db->fetchColumn("SELECT COUNT(*) FROM testimonials WHERE is_active = 1"),
        'blog_posts' => $db->fetchColumn("SELECT COUNT(*) FROM blog_posts WHERE status = 'published'"),
        'images' => $db->fetchColumn("SELECT COUNT(*) FROM service_images")
    ];
    
    // Son başvurular
    $recentLeads = $db->fetchAll(
        "SELECT * FROM contact_forms ORDER BY created_at DESC LIMIT 5"
    );
    
    // Son ziyaretçiler
    $recentVisitors = $db->fetchAll(
        "SELECT * FROM visitors ORDER BY visit_date DESC LIMIT 10"
    );
    
} catch (Exception $e) {
    error_log("Dashboard error: " . $e->getMessage());
    $stats = array_fill_keys(['services', 'leads', 'visitors_today', 'visitors_month', 'testimonials', 'blog_posts', 'images'], 0);
    $recentLeads = [];
    $recentVisitors = [];
}

include 'partials/header.php';
?>

<!-- Dashboard Content -->
<div class="p-6">
    <!-- Welcome Banner -->
    <div class="glass-card rounded-2xl p-6 mb-8">
        <div class="flex items-center justify-between">
            <div>
                <h1 class="text-2xl font-bold text-white">Hoş Geldiniz, <?= htmlspecialchars($admin['full_name']) ?>!</h1>
                <p class="text-white/60 mt-1">İzmir Tenteci Yönetim Paneli</p>
            </div>
            <div class="text-right">
                <p class="text-white/40 text-sm"><?= date('d F Y, l', strtotime('now')) ?></p>
                <p class="text-gold text-lg font-semibold"><?= date('H:i') ?></p>
            </div>
        </div>
    </div>
    
    <!-- Stats Grid -->
    <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6 mb-8">
        <!-- Hizmetler -->
        <div class="glass-card rounded-xl p-6">
            <div class="flex items-center justify-between">
                <div>
                    <p class="text-white/60 text-sm">Aktif Hizmetler</p>
                    <p class="text-3xl font-bold text-white mt-1"><?= $stats['services'] ?></p>
                </div>
                <div class="w-14 h-14 rounded-xl bg-blue-500/20 flex items-center justify-center">
                    <i data-lucide="briefcase" class="w-7 h-7 text-blue-400"></i>
                </div>
            </div>
            <a href="services.php" class="text-blue-400 text-sm mt-4 flex items-center gap-1 hover:text-blue-300">
                Yönet <i data-lucide="arrow-right" class="w-4 h-4"></i>
            </a>
        </div>
        
        <!-- Yeni Başvurular -->
        <div class="glass-card rounded-xl p-6">
            <div class="flex items-center justify-between">
                <div>
                    <p class="text-white/60 text-sm">Yeni Başvurular</p>
                    <p class="text-3xl font-bold text-white mt-1"><?= $stats['leads'] ?></p>
                </div>
                <div class="w-14 h-14 rounded-xl bg-gold/20 flex items-center justify-center">
                    <i data-lucide="message-square" class="w-7 h-7 text-gold"></i>
                </div>
            </div>
            <a href="leads.php" class="text-gold text-sm mt-4 flex items-center gap-1 hover:text-gold-light">
                Görüntüle <i data-lucide="arrow-right" class="w-4 h-4"></i>
            </a>
        </div>
        
        <!-- Bugünkü Ziyaretçi -->
        <div class="glass-card rounded-xl p-6">
            <div class="flex items-center justify-between">
                <div>
                    <p class="text-white/60 text-sm">Bugün Ziyaretçi</p>
                    <p class="text-3xl font-bold text-white mt-1"><?= $stats['visitors_today'] ?></p>
                </div>
                <div class="w-14 h-14 rounded-xl bg-green-500/20 flex items-center justify-center">
                    <i data-lucide="users" class="w-7 h-7 text-green-400"></i>
                </div>
            </div>
            <p class="text-green-400 text-sm mt-4">Bu ay: <?= $stats['visitors_month'] ?></p>
        </div>
        
        <!-- Görseller -->
        <div class="glass-card rounded-xl p-6">
            <div class="flex items-center justify-between">
                <div>
                    <p class="text-white/60 text-sm">Toplam Görsel</p>
                    <p class="text-3xl font-bold text-white mt-1"><?= $stats['images'] ?></p>
                </div>
                <div class="w-14 h-14 rounded-xl bg-purple-500/20 flex items-center justify-center">
                    <i data-lucide="image" class="w-7 h-7 text-purple-400"></i>
                </div>
            </div>
            <a href="gallery.php" class="text-purple-400 text-sm mt-4 flex items-center gap-1 hover:text-purple-300">
                Galeri <i data-lucide="arrow-right" class="w-4 h-4"></i>
            </a>
        </div>
    </div>
    
    <!-- Two Column Layout -->
    <div class="grid lg:grid-cols-2 gap-8">
        <!-- Son Başvurular -->
        <div class="glass-card rounded-2xl p-6">
            <div class="flex items-center justify-between mb-6">
                <h2 class="text-lg font-semibold text-white flex items-center gap-2">
                    <i data-lucide="inbox" class="w-5 h-5 text-gold"></i>
                    Son Başvurular
                </h2>
                <a href="leads.php" class="text-gold text-sm hover:text-gold-light">Tümünü Gör</a>
            </div>
            
            <?php if (empty($recentLeads)): ?>
            <div class="text-center py-8 text-white/40">
                <i data-lucide="inbox" class="w-12 h-12 mx-auto mb-3 opacity-50"></i>
                <p>Henüz başvuru yok</p>
            </div>
            <?php else: ?>
            <div class="space-y-4">
                <?php foreach ($recentLeads as $lead): ?>
                <div class="flex items-start gap-4 p-4 rounded-lg bg-dark-800/50 hover:bg-dark-800 transition-colors">
                    <div class="w-10 h-10 rounded-full bg-gold/20 flex items-center justify-center flex-shrink-0">
                        <span class="text-gold font-semibold"><?= strtoupper(mb_substr($lead['name'], 0, 1)) ?></span>
                    </div>
                    <div class="flex-1 min-w-0">
                        <div class="flex items-center justify-between">
                            <h3 class="text-white font-medium truncate"><?= htmlspecialchars($lead['name']) ?></h3>
                            <span class="text-white/40 text-xs"><?= date('d.m H:i', strtotime($lead['created_at'])) ?></span>
                        </div>
                        <p class="text-white/60 text-sm truncate"><?= htmlspecialchars($lead['phone']) ?></p>
                        <?php if ($lead['status'] === 'new'): ?>
                        <span class="inline-block mt-1 px-2 py-0.5 rounded text-xs bg-gold/20 text-gold">Yeni</span>
                        <?php endif; ?>
                    </div>
                </div>
                <?php endforeach; ?>
            </div>
            <?php endif; ?>
        </div>
        
        <!-- Hızlı Erişim -->
        <div class="glass-card rounded-2xl p-6">
            <h2 class="text-lg font-semibold text-white mb-6 flex items-center gap-2">
                <i data-lucide="zap" class="w-5 h-5 text-gold"></i>
                Hızlı Erişim
            </h2>
            
            <div class="grid grid-cols-2 gap-4">
                <a href="services.php?action=new" class="flex items-center gap-3 p-4 rounded-lg bg-dark-800/50 hover:bg-dark-700 transition-colors group">
                    <div class="w-10 h-10 rounded-lg bg-blue-500/20 flex items-center justify-center group-hover:bg-blue-500/30">
                        <i data-lucide="plus" class="w-5 h-5 text-blue-400"></i>
                    </div>
                    <span class="text-white text-sm">Yeni Hizmet</span>
                </a>
                
                <a href="gallery.php?action=upload" class="flex items-center gap-3 p-4 rounded-lg bg-dark-800/50 hover:bg-dark-700 transition-colors group">
                    <div class="w-10 h-10 rounded-lg bg-purple-500/20 flex items-center justify-center group-hover:bg-purple-500/30">
                        <i data-lucide="upload" class="w-5 h-5 text-purple-400"></i>
                    </div>
                    <span class="text-white text-sm">Görsel Yükle</span>
                </a>
                
                <a href="blog.php?action=new" class="flex items-center gap-3 p-4 rounded-lg bg-dark-800/50 hover:bg-dark-700 transition-colors group">
                    <div class="w-10 h-10 rounded-lg bg-green-500/20 flex items-center justify-center group-hover:bg-green-500/30">
                        <i data-lucide="file-text" class="w-5 h-5 text-green-400"></i>
                    </div>
                    <span class="text-white text-sm">Blog Yazısı</span>
                </a>
                
                <a href="settings.php" class="flex items-center gap-3 p-4 rounded-lg bg-dark-800/50 hover:bg-dark-700 transition-colors group">
                    <div class="w-10 h-10 rounded-lg bg-gold/20 flex items-center justify-center group-hover:bg-gold/30">
                        <i data-lucide="settings" class="w-5 h-5 text-gold"></i>
                    </div>
                    <span class="text-white text-sm">Ayarlar</span>
                </a>
            </div>
            
            <!-- System Health -->
            <div class="mt-6 pt-6 border-t border-white/10">
                <h3 class="text-white/60 text-sm mb-4">Sistem Durumu</h3>
                <div class="space-y-3">
                    <div class="flex items-center justify-between">
                        <span class="text-white/60 text-sm">PHP Sürümü</span>
                        <span class="text-green-400 text-sm"><?= PHP_VERSION ?></span>
                    </div>
                    <div class="flex items-center justify-between">
                        <span class="text-white/60 text-sm">Veritabanı</span>
                        <span class="text-green-400 text-sm flex items-center gap-1">
                            <span class="w-2 h-2 rounded-full bg-green-400"></span>
                            Bağlı
                        </span>
                    </div>
                    <div class="flex items-center justify-between">
                        <span class="text-white/60 text-sm">AI Asistan</span>
                        <?php $aiConfigured = (new AiAssistant())->isConfigured(); ?>
                        <span class="<?= $aiConfigured ? 'text-green-400' : 'text-yellow-400' ?> text-sm flex items-center gap-1">
                            <span class="w-2 h-2 rounded-full <?= $aiConfigured ? 'bg-green-400' : 'bg-yellow-400' ?>"></span>
                            <?= $aiConfigured ? 'Aktif' : 'Yapılandır' ?>
                        </span>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<?php include 'partials/footer.php'; ?>
