<?php
/**
 * AiAssistant - Yapay Zeka İçerik Asistanı
 * Aegean Dominance Admin Panel
 * 
 * Gemini/OpenAI API entegrasyonu, E-E-A-T uyumlu içerik üretimi
 */

declare(strict_types=1);

class AiAssistant
{
    private Database $db;
    private ?string $apiKey = null;
    private string $provider = 'gemini'; // gemini veya openai
    
    private const GEMINI_API_URL = 'https://generativelanguage.googleapis.com/v1beta/models/gemini-pro:generateContent';
    private const OPENAI_API_URL = 'https://api.openai.com/v1/chat/completions';
    
    public function __construct()
    {
        $this->db = Database::getInstance();
        $this->loadApiKey();
    }
    
    /**
     * API key yükle
     */
    private function loadApiKey(): void
    {
        try {
            // api_settings tablosu var mı kontrol et
            $tableExists = $this->db->fetchColumn(
                "SELECT COUNT(*) FROM information_schema.tables WHERE table_schema = DATABASE() AND table_name = 'api_settings'"
            );
            
            if (!$tableExists) {
                // Tablo yoksa sessizce devam et (henüz kurulmamış)
                return;
            }
            
            // Önce Gemini dene
            $geminiKey = $this->getDecryptedSetting('gemini_api_key');
            if ($geminiKey) {
                $this->apiKey = $geminiKey;
                $this->provider = 'gemini';
                return;
            }
            
            // OpenAI dene
            $openaiKey = $this->getDecryptedSetting('openai_api_key');
            if ($openaiKey) {
                $this->apiKey = $openaiKey;
                $this->provider = 'openai';
            }
            
        } catch (Exception $e) {
            error_log("AiAssistant::loadApiKey error: " . $e->getMessage());
        }
    }
    
    /**
     * Şifreli ayar al
     */
    private function getDecryptedSetting(string $key): ?string
    {
        $result = $this->db->fetchOne(
            "SELECT setting_value, is_encrypted FROM api_settings WHERE setting_key = ?",
            [$key]
        );
        
        if (!$result) return null;
        
        if ($result['is_encrypted']) {
            return $this->decrypt($result['setting_value']);
        }
        
        return $result['setting_value'];
    }
    
    // =========================================================================
    // İÇERİK ÜRETİMİ
    // =========================================================================
    
    /**
     * Ürün açıklaması oluştur
     */
    public function generateProductDescription(string $title, string $category = ''): array
    {
        if (!$this->apiKey) {
            return ['success' => false, 'error' => 'AI API yapılandırılmamış.'];
        }
        
        $prompt = $this->buildProductPrompt($title, $category);
        
        return $this->generate($prompt);
    }
    
    /**
     * Ürün açıklaması prompt'u
     */
    private function buildProductPrompt(string $title, string $category): string
    {
        return <<<PROMPT
Sen profesyonel bir SEO içerik yazarısın. İzmir Tenteci firması için "{$title}" ürünü hakkında bir açıklama yaz.

Kurallar:
1. E-E-A-T (Experience, Expertise, Authoritativeness, Trustworthiness) prensiplerine uygun yaz.
2. İzmir ve Ege Bölgesi'ne özgü yerel referanslar kullan.
3. 35 yıllık tecrübeyi vurgula.
4. Teknik detayları sade bir dille anlat.
5. 300-500 kelime arasında olsun.
6. H2 ve H3 alt başlıklar kullan.
7. Maddeli liste içersin.
8. Anahtar kelime yığılması yapma.
9. Call-to-action içersin: "Ücretsiz keşif için hemen arayın."
10. Türkçe yaz, profesyonel ama samimi bir üslup kullan.

Kategori: {$category}
Ürün: {$title}

Çıktı formatı:
- Sadece HTML içerik ver (<h2>, <h3>, <p>, <ul>, <li> etiketleri kullan)
- Açıklama veya yorum ekleme
PROMPT;
    }
    
    /**
     * Meta description oluştur
     */
    public function generateMetaDescription(string $title): array
    {
        if (!$this->apiKey) {
            return ['success' => false, 'error' => 'AI API yapılandırılmamış.'];
        }
        
        $prompt = <<<PROMPT
"{$title}" için SEO uyumlu meta description yaz.

Kurallar:
- 140-160 karakter arası
- İzmir Tenteci firması için
- CTA içersin (Örn: "Hemen arayın" veya "Ücretsiz teklif alın")
- Ana anahtar kelimeyi en başta kullan
- Telefon numarasını koyma

Sadece meta description'ı yaz, başka bir şey ekleme.
PROMPT;
        
        return $this->generate($prompt);
    }
    
    /**
     * Şehir landing page içeriği oluştur
     */
    public function generateCityContent(string $cityName, string $baseContent): array
    {
        if (!$this->apiKey) {
            return ['success' => false, 'error' => 'AI API yapılandırılmamış.'];
        }
        
        $prompt = <<<PROMPT
Aşağıdaki içeriği "{$cityName}" şehri için uyarla. Şehir adını ve yerel referansları değiştir, ama ana içerik yapısını koru.

Kurallar:
1. {$cityName} ve çevresindeki ilçelere referans ver
2. Yerel özellikler ekle (iklim, mimari vb.)
3. "İzmir Tenteci {$cityName} şubesi" vurgusunu yap
4. SEO için {$cityName} anahtar kelimesini doğal kullan
5. HTML formatında çıktı ver

Orijinal içerik:
{$baseContent}
PROMPT;
        
        return $this->generate($prompt);
    }
    
    // =========================================================================
    // API ÇAĞRILARI
    // =========================================================================
    
    /**
     * AI isteği gönder
     */
    private function generate(string $prompt): array
    {
        try {
            if ($this->provider === 'gemini') {
                return $this->callGemini($prompt);
            } else {
                return $this->callOpenAI($prompt);
            }
        } catch (Exception $e) {
            error_log("AiAssistant::generate error: " . $e->getMessage());
            return ['success' => false, 'error' => $e->getMessage()];
        }
    }
    
    /**
     * Gemini API çağrısı
     */
    private function callGemini(string $prompt): array
    {
        $url = self::GEMINI_API_URL . '?key=' . $this->apiKey;
        
        $data = [
            'contents' => [
                ['parts' => [['text' => $prompt]]]
            ],
            'generationConfig' => [
                'temperature' => 0.7,
                'maxOutputTokens' => 2048
            ]
        ];
        
        $response = $this->curlRequest($url, [
            'method' => 'POST',
            'headers' => ['Content-Type: application/json'],
            'body' => json_encode($data)
        ]);
        
        if ($response['http_code'] !== 200) {
            $error = json_decode($response['body'], true);
            throw new Exception($error['error']['message'] ?? 'Gemini API hatası');
        }
        
        $result = json_decode($response['body'], true);
        $text = $result['candidates'][0]['content']['parts'][0]['text'] ?? '';
        
        return [
            'success' => true,
            'content' => $text,
            'provider' => 'gemini'
        ];
    }
    
    /**
     * OpenAI API çağrısı
     */
    private function callOpenAI(string $prompt): array
    {
        $data = [
            'model' => 'gpt-4-turbo-preview',
            'messages' => [
                ['role' => 'system', 'content' => 'Sen profesyonel bir Türkçe SEO içerik yazarısın.'],
                ['role' => 'user', 'content' => $prompt]
            ],
            'temperature' => 0.7,
            'max_tokens' => 2048
        ];
        
        $response = $this->curlRequest(self::OPENAI_API_URL, [
            'method' => 'POST',
            'headers' => [
                'Content-Type: application/json',
                'Authorization: Bearer ' . $this->apiKey
            ],
            'body' => json_encode($data)
        ]);
        
        if ($response['http_code'] !== 200) {
            $error = json_decode($response['body'], true);
            throw new Exception($error['error']['message'] ?? 'OpenAI API hatası');
        }
        
        $result = json_decode($response['body'], true);
        $text = $result['choices'][0]['message']['content'] ?? '';
        
        return [
            'success' => true,
            'content' => $text,
            'provider' => 'openai'
        ];
    }
    
    // =========================================================================
    // HELPERS
    // =========================================================================
    
    private function curlRequest(string $url, array $options = []): array
    {
        $ch = curl_init();
        
        curl_setopt_array($ch, [
            CURLOPT_URL => $url,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_TIMEOUT => 60,
            CURLOPT_SSL_VERIFYPEER => true
        ]);
        
        if (($options['method'] ?? 'GET') === 'POST') {
            curl_setopt($ch, CURLOPT_POST, true);
            if (isset($options['body'])) {
                curl_setopt($ch, CURLOPT_POSTFIELDS, $options['body']);
            }
        }
        
        if (isset($options['headers'])) {
            curl_setopt($ch, CURLOPT_HTTPHEADER, $options['headers']);
        }
        
        $body = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $error = curl_error($ch);
        
        curl_close($ch);
        
        if ($error) {
            throw new Exception("cURL Error: {$error}");
        }
        
        return ['http_code' => $httpCode, 'body' => $body];
    }
    
    private function decrypt(string $data): ?string
    {
        try {
            if (empty($data)) {
                return null;
            }
            $key = hash('sha256', 'AegeanDominance2025SecretKey!');
            $decoded = base64_decode($data);
            if ($decoded === false || strlen($decoded) < 16) {
                return null;
            }
            $iv = substr($decoded, 0, 16);
            $encrypted = substr($decoded, 16);
            $result = openssl_decrypt($encrypted, 'AES-256-CBC', $key, 0, $iv);
            return $result !== false ? $result : null;
        } catch (Exception $e) {
            error_log("AiAssistant::decrypt error: " . $e->getMessage());
            return null;
        }
    }
    
    /**
     * API durumu kontrol
     */
    public function isConfigured(): bool
    {
        return $this->apiKey !== null;
    }
    
    /**
     * Mevcut provider
     */
    public function getProvider(): string
    {
        return $this->provider;
    }
}

/**
 * AiAssistant helper
 */
function ai(): AiAssistant
{
    static $instance = null;
    if ($instance === null) {
        $instance = new AiAssistant();
    }
    return $instance;
}
