<?php
/**
 * Conductor - SEO Orkestra Şefi
 * Aegean Dominance - İzmir Tenteci V2
 * 
 * SeoManager, SeoAuditor ve AiAssistant sınıflarını yöneterek
 * siteyi Google'da 1. sıraya taşımak için otonom iyileştirmeler yapar.
 */

declare(strict_types=1);

class Conductor
{
    private Database $db;
    private SeoManager $seo;
    private SeoAuditor $auditor;
    private AiAssistant $ai;
    private array $logs = [];

    public function __construct()
    {
        $this->db = Database::getInstance();
        $this->seo = new SeoManager();
        $this->auditor = new SeoAuditor();
        $this->ai = new AiAssistant();
    }

    /**
     * Tüm siteyi tara ve zayıf sayfaları tespit et
     */
    public function conductFullAudit(): array
    {
        $report = [
            'services' => $this->auditServices(),
            'blogs' => $this->auditBlogs(),
            'cities' => $this->auditCities(),
            'timestamp' => date('Y-m-d H:i:s')
        ];

        $this->log("Saha denetimi tamamlandı.");
        return $report;
    }

    /**
     * Hizmetleri denetle
     */
    private function auditServices(): array
    {
        $services = $this->db->fetchAll("SELECT * FROM services WHERE is_active = 1");
        $results = [];

        foreach ($services as $service) {
            $analysis = $this->auditor->analyze(
                $service['name'],
                $service['description'] ?? '',
                $service['meta_description'] ?? ''
            );

            $results[] = [
                'id' => $service['id'],
                'name' => $service['name'],
                'score' => $analysis['score'],
                'needs_healing' => $analysis['score'] < 85,
                'issues' => array_merge($analysis['errors'], $analysis['warnings'])
            ];
        }

        return $results;
    }

    /**
     * Zayıf bir hizmet sayfasını otomatik iyileştir
     */
    public function healService(int $serviceId): array
    {
        $service = $this->db->fetchOne("SELECT * FROM services WHERE id = ?", [$serviceId]);
        if (!$service) return ['success' => false, 'error' => 'Hizmet bulunamadı.'];

        $this->log("İyileştirme başladı: " . $service['name']);

        $updates = [];

        // 1. Meta Description eksik veya zayıfsa AI ile üret
        if (empty($service['meta_description']) || mb_strlen($service['meta_description']) < 100) {
            $aiMeta = $this->ai->generateMetaDescription($service['name']);
            if ($aiMeta['success']) {
                $updates['meta_description'] = $aiMeta['content'];
                $this->log("Meta açıklaması AI ile güncellendi.");
            }
        }

        // 2. İçerik zayıfsa (300 kelime altı) AI ile zenginleştir
        $wordCount = str_word_count(strip_tags($service['description'] ?? ''), 0, 'ğüşıöçĞÜŞİÖÇ');
        if ($wordCount < 300) {
            $aiContent = $this->ai->generateProductDescription($service['name'], $service['category'] ?? '');
            if ($aiContent['success']) {
                $updates['description'] = $aiContent['content'];
                $this->log("İçerik AI ile zenginleştirildi (E-E-A-T uyumlu).");
            }
        }

        // 3. Meta Title kontrolü
        if (empty($service['meta_title'])) {
            $updates['meta_title'] = $this->auditor->suggestTitle($service['name'], $service['category'] ?? '');
        }

        if (!empty($updates)) {
            $fields = implode(' = ?, ', array_keys($updates)) . ' = ?';
            $params = array_values($updates);
            $params[] = $serviceId;
            
            $this->db->execute("UPDATE services SET {$fields} WHERE id = ?", $params);
            return ['success' => true, 'updated_fields' => array_keys($updates)];
        }

        return ['success' => true, 'message' => 'Sayfa zaten optimize edilmiş.'];
    }

    /**
     * Şehir sayfalarını denetle ve gerekirse içerik üret
     */
    private function auditCities(): array
    {
        $cities = $this->db->fetchAll("SELECT * FROM cities WHERE is_active = 1");
        $results = [];

        foreach ($cities as $city) {
            if (empty($city['content'])) {
                $results[] = [
                    'id' => $city['id'],
                    'name' => $city['name'],
                    'status' => 'empty',
                    'needs_content' => true
                ];
            }
        }

        return $results;
    }

    /**
     * Boş bir şehir sayfasını doldur
     */
    public function healCity(int $cityId): array
    {
        $city = $this->db->fetchOne("SELECT * FROM cities WHERE id = ?", [$cityId]);
        if (!$city) return ['success' => false, 'error' => 'Şehir bulunamadı.'];

        // Temel içerik olarak "Bioklimatik Pergola" içeriğini baz alabiliriz
        $baseService = $this->db->fetchOne("SELECT description FROM services WHERE slug = 'bioklimatik-pergola'");
        $baseContent = $baseService['description'] ?? 'İzmir Tente ve Pergola sistemleri konusunda uzman kadromuzla hizmetinizdeyiz.';

        $aiContent = $this->ai->generateCityContent($city['name'], $baseContent);
        
        if ($aiContent['success']) {
            $metaTitle = "{$city['name']} Tente ve Pergola Sistemleri | İzmir Tenteci";
            $metaDesc = "{$city['name']} ve çevresinde profesyonel tente, pergola ve cam balkon çözümleri. Ücretsiz keşif için hemen arayın.";

            $this->db->execute(
                "UPDATE cities SET content = ?, meta_title = ?, meta_description = ? WHERE id = ?",
                [$aiContent['content'], $metaTitle, $metaDesc, $cityId]
            );
            
            $this->log("{$city['name']} sayfası yerel içerikle dolduruldu.");
            return ['success' => true];
        }

        return ['success' => false, 'error' => 'AI içerik üretemedi.'];
    }

    /**
     * Blogları denetle
     */
    private function auditBlogs(): array
    {
        $blogs = $this->db->fetchAll("SELECT * FROM blog_posts WHERE status = 'published'");
        $results = [];

        foreach ($blogs as $blog) {
            $analysis = $this->auditor->analyze($blog['title'], $blog['content'], $blog['meta_description'] ?? '');
            $results[] = [
                'id' => $blog['id'],
                'title' => $blog['title'],
                'score' => $analysis['score'],
                'needs_healing' => $analysis['score'] < 80
            ];
        }

        return $results;
    }

    /**
     * Oto-Pilot durumunu getir
     */
    public function isAutoPilotEnabled(): bool
    {
        $val = $this->db->fetchColumn("SELECT setting_value FROM settings WHERE setting_key = 'conductor_autopilot'");
        return $val === '1' || $val === 'true';
    }

    /**
     * Oto-Pilot durumunu değiştir
     */
    public function setAutoPilot(bool $enabled): void
    {
        // Ayar yoksa oluştur, varsa güncelle
        $exists = $this->db->fetchColumn("SELECT COUNT(*) FROM settings WHERE setting_key = 'conductor_autopilot'");
        
        if ($exists) {
            $this->db->execute("UPDATE settings SET setting_value = ? WHERE setting_key = 'conductor_autopilot'", [$enabled ? '1' : '0']);
        } else {
            $this->db->execute(
                "INSERT INTO settings (setting_key, setting_value, category, description) VALUES (?, ?, ?, ?)",
                ['conductor_autopilot', $enabled ? '1' : '0', 'system', 'Conductor Oto-Pilot Modu']
            );
        }
        
        $this->log("Oto-Pilot modu " . ($enabled ? "açıldı" : "kapatıldı") . ".");
    }

    /**
     * Cron Job veya "Hepsini Düzelt" butonu için toplu işlem
     * Eğer $force=true ise ayara bakmaksızın çalışır.
     */
    public function runAutoPilotRoutine(bool $force = false): array
    {
        if (!$force && !$this->isAutoPilotEnabled()) {
            return ['success' => false, 'message' => 'Oto-Pilot kapalı.'];
        }

        $this->log("Oto-Pilot rutini başladı...");
        $audit = $this->conductFullAudit();
        $actions = [];

        // 1. Hizmetleri İyileştir (Puanı 85 altı olanlar)
        foreach ($audit['services'] as $service) {
            if ($service['needs_healing']) {
                $result = $this->healService((int)$service['id']);
                if ($result['success']) {
                    $actions[] = "Hizmet İyileştirildi: {$service['name']}";
                }
            }
        }

        // 2. Şehir Sayfalarını Oluştur (İçeriği boş olanlar)
        foreach ($audit['cities'] as $city) {
            if ($city['needs_content']) {
                $result = $this->healCity((int)$city['id']);
                if ($result['success']) {
                    $actions[] = "Şehir Sayfası Oluşturuldu: {$city['name']}";
                }
            }
        }

        // 3. Blogları Kontrol Et (Opsiyonel genişletilebilir)
        
        $this->log("Oto-Pilot rutini tamamlandı. " . count($actions) . " işlem yapıldı.");
        
        return [
            'success' => true,
            'processed_count' => count($actions),
            'actions' => $actions
        ];
    }

    /**
     * Log kaydı
     */
    private function log(string $message): void
    {
        $this->logs[] = "[" . date('H:i:s') . "] " . $message;
    }

    public function getLogs(): array
    {
        return $this->logs;
    }
}
