<?php
/**
 * SeoAuditor - SEO Analiz ve İçerik Denetleme Sınıfı
 * Aegean Dominance Admin Panel
 * 
 * Semantik analiz, anahtar kelime kontrolü, E-E-A-T uyumluluk
 */

declare(strict_types=1);

class SeoAuditor
{
    private array $warnings = [];
    private array $errors = [];
    private array $suggestions = [];
    
    // Anahtar kelimeler
    private array $primaryKeywords = [
        'tente', 'pergola', 'bioklimatik', 'cam balkon', 'zip perde',
        'gölgelendirme', 'izmir', 'mafsallı', 'motorlu'
    ];
    
    // =========================================================================
    // İÇERİK ANALİZİ
    // =========================================================================
    
    /**
     * İçerik analizi yap
     */
    public function analyze(string $title, string $content, string $metaDescription = ''): array
    {
        $this->warnings = [];
        $this->errors = [];
        $this->suggestions = [];
        
        try {
            // Başlık analizi
            $this->analyzeTitle($title);
            
            // Meta description analizi
            $this->analyzeMetaDescription($metaDescription);
            
            // İçerik analizi
            $this->analyzeContent($content, $title);
            
            // Skor hesapla
            $score = $this->calculateScore();
            
            return [
                'success' => true,
                'score' => $score,
                'grade' => $this->getGrade($score),
                'errors' => $this->errors,
                'warnings' => $this->warnings,
                'suggestions' => $this->suggestions
            ];
            
        } catch (Exception $e) {
            error_log("SeoAuditor::analyze error: " . $e->getMessage());
            return [
                'success' => false,
                'error' => 'Analiz sırasında hata oluştu.'
            ];
        }
    }
    
    /**
     * Başlık analizi
     */
    private function analyzeTitle(string $title): void
    {
        $length = mb_strlen($title);
        
        if ($length < 30) {
            $this->warnings[] = "Başlık çok kısa ({$length} karakter). Minimum 30 karakter önerilir.";
        } elseif ($length > 60) {
            $this->warnings[] = "Başlık çok uzun ({$length} karakter). Google 60 karakterden sonrasını kesebilir.";
        }
        
        // Anahtar kelime kontrolü
        $hasKeyword = false;
        foreach ($this->primaryKeywords as $keyword) {
            if (mb_stripos($title, $keyword) !== false) {
                $hasKeyword = true;
                break;
            }
        }
        
        if (!$hasKeyword) {
            $this->warnings[] = "Başlıkta ana anahtar kelimelerden biri bulunmuyor.";
        }
        
        // Sayı veya yıl kontrolü
        if (preg_match('/\d{4}/', $title)) {
            $this->suggestions[] = "✓ Başlıkta yıl kullanımı CTR'ı artırabilir.";
        }
    }
    
    /**
     * Meta description analizi
     */
    private function analyzeMetaDescription(string $description): void
    {
        if (empty($description)) {
            $this->errors[] = "Meta açıklama boş. Google arama sonuçlarında gösterilecek açıklama yazın.";
            return;
        }
        
        $length = mb_strlen($description);
        
        if ($length < 120) {
            $this->warnings[] = "Meta açıklama çok kısa ({$length} karakter). 120-160 karakter önerilir.";
        } elseif ($length > 160) {
            $this->warnings[] = "Meta açıklama çok uzun ({$length} karakter). 160 karakterden sonrası kesilir.";
        }
        
        // CTA kontrolü
        $ctaWords = ['hemen', 'ücretsiz', 'teklif', 'arayın', 'keşif', 'fiyat'];
        $hasCta = false;
        foreach ($ctaWords as $cta) {
            if (mb_stripos($description, $cta) !== false) {
                $hasCta = true;
                break;
            }
        }
        
        if (!$hasCta) {
            $this->suggestions[] = "Meta açıklamaya bir CTA (call-to-action) ekleyin: 'Hemen arayın', 'Ücretsiz teklif' vb.";
        }
    }
    
    /**
     * İçerik analizi
     */
    private function analyzeContent(string $content, string $title): void
    {
        // HTML temizle
        $plainText = strip_tags($content);
        $wordCount = str_word_count($plainText, 0, 'ğüşıöçĞÜŞİÖÇ');
        
        // Kelime sayısı
        if ($wordCount < 300) {
            $this->errors[] = "İçerik çok kısa ({$wordCount} kelime). Minimum 300 kelime önerilir.";
        } elseif ($wordCount < 500) {
            $this->warnings[] = "İçerik orta uzunlukta ({$wordCount} kelime). 500+ kelime daha iyi sıralama sağlar.";
        } else {
            $this->suggestions[] = "✓ İçerik uzunluğu yeterli ({$wordCount} kelime).";
        }
        
        // Alt başlık kontrolü
        $h2Count = preg_match_all('/<h2[^>]*>/i', $content);
        $h3Count = preg_match_all('/<h3[^>]*>/i', $content);
        
        if ($h2Count == 0) {
            $this->errors[] = "İçerikte H2 alt başlık yok. Yapısal SEO için alt başlıklar ekleyin.";
        }
        
        if ($wordCount > 500 && ($h2Count + $h3Count) < 3) {
            $this->warnings[] = "Uzun içerik için daha fazla alt başlık önerilir.";
        }
        
        // Anahtar kelime yoğunluğu
        $this->analyzeKeywordDensity($plainText);
        
        // Liste kontrolü
        if (strpos($content, '<ul') === false && strpos($content, '<ol') === false) {
            $this->suggestions[] = "İçeriğe maddeli veya numaralı liste eklemek okunabilirliği artırır.";
        }
        
        // Görsel kontrolü
        $imgCount = preg_match_all('/<img[^>]*>/i', $content);
        if ($imgCount == 0) {
            $this->warnings[] = "İçerikte görsel yok. Her 300 kelimede en az 1 görsel önerilir.";
        } else {
            // Alt tag kontrolü
            preg_match_all('/<img[^>]*alt\s*=\s*["\']([^"\']*)["\'][^>]*>/i', $content, $altMatches);
            $emptyAlts = 0;
            foreach ($altMatches[1] as $alt) {
                if (empty(trim($alt))) {
                    $emptyAlts++;
                }
            }
            if ($emptyAlts > 0) {
                $this->errors[] = "{$emptyAlts} görselde alt etiketi boş. SEO için alt etiketleri doldurun.";
            }
        }
        
        // İç link kontrolü
        if (strpos($content, 'href=') === false) {
            $this->suggestions[] = "İç link eklenmemiş. Diğer sayfalara link vererek site yapısını güçlendirin.";
        }
    }
    
    /**
     * Anahtar kelime yoğunluğu analizi
     */
    private function analyzeKeywordDensity(string $text): void
    {
        $text = mb_strtolower($text);
        $wordCount = str_word_count($text, 0, 'ğüşıöçĞÜŞİÖÇ');
        
        if ($wordCount == 0) return;
        
        foreach ($this->primaryKeywords as $keyword) {
            $count = mb_substr_count($text, $keyword);
            $density = ($count / $wordCount) * 100;
            
            if ($density > 3) {
                $this->errors[] = "'{$keyword}' kelimesi çok fazla kullanılmış (%{$density}). Anahtar kelime yığılması riski!";
            }
        }
    }
    
    /**
     * SEO skoru hesapla
     */
    private function calculateScore(): int
    {
        $score = 100;
        
        // Her hata için -15
        $score -= count($this->errors) * 15;
        
        // Her uyarı için -8
        $score -= count($this->warnings) * 8;
        
        // Her öneri için +3 (en fazla 15)
        $suggestionBonus = min(count($this->suggestions) * 3, 15);
        $score += $suggestionBonus;
        
        return max(0, min(100, $score));
    }
    
    /**
     * Skor notu
     */
    private function getGrade(int $score): string
    {
        if ($score >= 90) return 'A+';
        if ($score >= 80) return 'A';
        if ($score >= 70) return 'B';
        if ($score >= 60) return 'C';
        if ($score >= 50) return 'D';
        return 'F';
    }
    
    // =========================================================================
    // SLUG OTOMATİK OLUŞTURMA
    // =========================================================================
    
    /**
     * SEO dostu slug oluştur
     */
    public static function generateSlug(string $text): string
    {
        // Türkçe karakterleri değiştir
        $replacements = [
            'ğ' => 'g', 'Ğ' => 'g',
            'ü' => 'u', 'Ü' => 'u',
            'ş' => 's', 'Ş' => 's',
            'ı' => 'i', 'İ' => 'i',
            'ö' => 'o', 'Ö' => 'o',
            'ç' => 'c', 'Ç' => 'c'
        ];
        
        $text = strtr($text, $replacements);
        $text = mb_strtolower($text);
        $text = preg_replace('/[^a-z0-9\s-]/', '', $text);
        $text = preg_replace('/[\s-]+/', '-', $text);
        $text = trim($text, '-');
        
        return $text;
    }
    
    // =========================================================================
    // META ÖNERİLERİ
    // =========================================================================
    
    /**
     * Başlık önerisi üret
     */
    public function suggestTitle(string $productName, string $category = ''): string
    {
        $year = date('Y');
        $templates = [
            "İzmir {$productName} Fiyatları ve Modelleri {$year}",
            "{$productName} Sistemleri - Profesyonel Montaj | İzmir Tenteci",
            "{$category} {$productName} - Ücretsiz Keşif ve Teklif"
        ];
        
        return $templates[array_rand($templates)];
    }
    
    /**
     * Meta açıklama önerisi üret
     */
    public function suggestDescription(string $productName, string $category = ''): string
    {
        return "İzmir ve Ege Bölgesinde profesyonel {$productName} hizmeti. " .
               "{$category} kategorisinde kaliteli çözümler. " .
               "5 yıl garanti, ücretsiz keşif. ☎ 0536 277 17 09";
    }
}

/**
 * SeoAuditor helper
 */
function seoAudit(): SeoAuditor
{
    return new SeoAuditor();
}
