<?php
/**
 * SeoManager - Gelişmiş SEO Yönetimi
 * Yapı Dekorasyon V2
 * 
 * Dinamik meta etiketleri, canonical URL'ler ve JSON-LD Schema.org yapıları
 */

declare(strict_types=1);

class SeoManager
{
    private Database $db;
    private string $pageType = 'home';
    private array $pageData = [];
    private string $currentUrl;
    private string $baseUrl;

    // Firma Bilgileri (Schema için)
    private array $businessInfo = [
        'name' => 'Yapı Dekorasyon',
        'legalName' => 'Yapı Dekorasyon İç Mimarlık ve Tadilat',
        'description' => 'İzmir ve Ege Bölgesinde profesyonel anahtar teslim tadilat, iç mimarlık, dekorasyon ve yenileme hizmetleri.',
        'foundingDate' => '2010',
        'telephone' => '+90 232 277 11 00',
        'email' => 'info@yapidekorasyon.com',
        'address' => [
            'streetAddress' => 'Mimarlık Cad. No:123',
            'addressLocality' => 'Konak',
            'addressRegion' => 'İzmir',
            'postalCode' => '35000',
            'addressCountry' => 'TR'
        ],
        'geo' => [
            'latitude' => '38.4192',
            'longitude' => '27.1287'
        ],
        'openingHours' => [
            'Mo-Sa 09:00-18:00'
        ],
        'priceRange' => '₺₺₺',
        'logo' => '/assets/images/logo.png',
        'image' => '/assets/images/hero-bg.jpg',
        'sameAs' => [
            'https://facebook.com/yapidekorasyon',
            'https://instagram.com/yapidekorasyon'
        ],
        'areaServed' => ['İzmir', 'Manisa', 'Aydın', 'Muğla']
    ];

    public function __construct()
    {
        $this->db = Database::getInstance();
        $this->baseUrl = defined('BASE_URL') ? BASE_URL : 'https://yapidekorasyon.com';
        $this->currentUrl = $this->getCurrentUrl();
    }

    /**
     * Sayfa tipini ve verisini ayarla
     */
    public function setPage(string $type, array $data = []): self
    {
        $this->pageType = $type;
        $this->pageData = $data;
        return $this;
    }

    /**
     * Mevcut temiz URL'yi al
     */
    private function getCurrentUrl(): string
    {
        $protocol = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ? 'https' : 'http';
        $host = $_SERVER['HTTP_HOST'] ?? 'yapidekorasyon.com';
        $uri = strtok($_SERVER['REQUEST_URI'] ?? '/', '?'); // Query string'i temizle
        
        // Base URL path kısmını (örn: /v2/public) kontrol et
        $basePath = parse_url($this->baseUrl, PHP_URL_PATH) ?? '';
        if ($basePath && strpos($uri, $basePath) !== 0) {
           // $uri zaten full path içeriyorsa dokunma
        }
        
        return $protocol . '://' . $host . $uri;
    }

    // =========================================================================
    // META ETİKETLERİ
    // =========================================================================

    /**
     * Dinamik meta etiketleri üret
     */
    public function generateMeta(): string
    {
        $title = $this->generateTitle();
        $description = $this->generateDescription();
        $keywords = $this->generateKeywords();

        $html = "<!-- SEO Meta Tags - Generated by SeoManager -->\n";
        $html .= '<title>' . htmlspecialchars($title) . "</title>\n";
        $html .= '<meta name="description" content="' . htmlspecialchars($description) . "\">\n";
        $html .= '<meta name="keywords" content="' . htmlspecialchars($keywords) . "\">\n";
        $html .= '<meta name="author" content="Yapı Dekorasyon">' . "\n";
        $html .= '<meta name="robots" content="index, follow, max-snippet:-1, max-image-preview:large, max-video-preview:-1">' . "\n";
        
        // Open Graph
        $html .= $this->generateOpenGraph($title, $description);
        
        // Twitter Card
        $html .= $this->generateTwitterCard($title, $description);
        
        return $html;
    }

    /**
     * Sayfa başlığı üret
     */
    private function generateTitle(): string
    {
        $firmaSuffix = ' | Yapı Dekorasyon';
        
        switch ($this->pageType) {
            case 'home':
                return 'Yapı Dekorasyon | İzmir Anahtar Teslim Tadilat & İç Mimarlık';
            
            case 'service':
                $serviceName = $this->pageData['name'] ?? 'Hizmet';
                return "İzmir {$serviceName} - Tadilat & Tasarım 2025{$firmaSuffix}";
            
            case 'services':
                return 'Tadilat ve Dekorasyon Hizmetlerimiz - Profesyonel Çözümler' . $firmaSuffix;
            
            case 'about':
                return 'Hakkımızda - Profesyonel Mimari Tasarım ve Uygulama' . $firmaSuffix;
            
            case 'contact':
                return 'İletişim - Ücretsiz Keşif ve Fiyat Teklifi' . $firmaSuffix;
            
            case 'gallery':
                return 'Proje Galerisi - Tamamlanan Tadilat Projeleri' . $firmaSuffix;
            
            case 'quote':
                return 'Ücretsiz Teklif Al - Hayalinizdeki Evi Tasarlayalım' . $firmaSuffix;
            
            case 'references':
                return 'Müşteri Yorumları ve Referanslar' . $firmaSuffix;
            
            case 'city':
                $cityName = $this->pageData['name'] ?? 'İzmir';
                return "{$cityName} Tadilat ve Dekorasyon Hizmetleri{$firmaSuffix}";
            
            case 'blog':
                return 'Blog - Tadilat, Dekorasyon ve Mimarlık Rehberi' . $firmaSuffix;
            
            case 'blog-detail':
                $postTitle = $this->pageData['title'] ?? 'Blog Yazısı';
                return $postTitle . $firmaSuffix;
            
            case 'faq':
                return 'Sıkça Sorulan Sorular - Tadilat Hakkında Her Şey' . $firmaSuffix;

            case '404':
                return 'Sayfa Bulunamadı' . $firmaSuffix;
            
            default:
                return 'Yapı Dekorasyon | Profesyonel İç Mimarlık ve Tadilat';
        }
    }

    /**
     * Meta açıklama üret
     */
    private function generateDescription(): string
    {
        switch ($this->pageType) {
            case 'home':
                return 'İzmir\'de profesyonel anahtar teslim tadilat, iç mimarlık, ev yenileme, banyo ve mutfak tasarımı hizmetleri. Ücretsiz keşif ve 3D tasarım desteği.';
            
            case 'service':
                $name = $this->pageData['name'] ?? 'Tadilat';
                $desc = $this->pageData['short_description'] ?? '';
                // Eğer kısa açıklama yoksa veya çok kısaysa varsayılan bir metin oluştur
                if (mb_strlen($desc) < 10) {
                     return "İzmir {$name} hizmetleri. Profesyonel tasarım, kaliteli malzeme ve usta işçilik ile {$name} çözümleri. Ücretsiz keşif için hemen arayın.";
                }
                return $desc;
            
            case 'services':
                return 'Komple ev tadilatı, banyo yenileme, mutfak tasarımı, seramik döşeme ve daha fazlası. Tüm tadilat ve dekorasyon ihtiyaçlarınız için profesyonel çözümler.';
            
            case 'about':
                return 'Yapı Dekorasyon olarak İzmir ve çevresinde yılların tecrübesiyle iç mimarlık ve tadilat hizmetleri sunuyoruz. Kalite ve güvenin adresi.';
            
            case 'contact':
                return 'Yapı Dekorasyon iletişim bilgileri. İzmir ofisimizden randevu alın veya ücretsiz keşif hizmetimizden yararlanın. Tel: 0232 277 11 00';
            
            case 'gallery':
                return 'Tamamladığımız ev, ofis, banyo ve mutfak tadilat projelerinden görseller. Öncesi ve sonrası proje fotoğrafları.';
            
            case 'city':
                $cityName = $this->pageData['name'] ?? 'İzmir';
                return "{$cityName} genelinde profesyonel tadilat ve dekorasyon hizmetleri. Anahtar teslim projeler ve iç mimarlık desteği.";
            
            default:
                return 'İzmir ve Ege Bölgesinde profesyonel anahtar teslim tadilat, iç mimarlık ve dekorasyon hizmetleri.';
        }
    }

    /**
     * Anahtar kelimeler üret
     */
    private function generateKeywords(): string
    {
        $baseKeywords = 'izmir tadilat, ev yenileme, iç mimarlık, dekorasyon, anahtar teslim tadilat';
        
        switch ($this->pageType) {
            case 'service':
                $name = $this->pageData['name'] ?? '';
                $slug = $this->pageData['slug'] ?? '';
                return "izmir {$slug}, {$name} fiyatları, {$name} modelleri, {$baseKeywords}";
            
            case 'city':
                $cityName = strtolower($this->pageData['name'] ?? 'izmir');
                return "{$cityName} tadilat, {$cityName} dekorasyon, {$cityName} boya badana, {$cityName} iç mimar";
            
            default:
                return $baseKeywords;
        }
    }

    /**
     * Open Graph etiketleri
     */
    private function generateOpenGraph(string $title, string $description): string
    {
        $image = $this->pageData['featured_image'] ?? '/assets/images/hero-bg.jpg';
        
        // Image URL'ini düzelt (tam URL değilse base ekle)
        if (strpos($image, 'http') !== 0) {
            $image = rtrim($this->baseUrl, '/') . '/' . ltrim($image, '/');
        }

        $html = "\n<!-- Open Graph -->\n";
        $html .= '<meta property="og:type" content="website">' . "\n";
        $html .= '<meta property="og:title" content="' . htmlspecialchars($title) . '">' . "\n";
        $html .= '<meta property="og:description" content="' . htmlspecialchars($description) . '">' . "\n";
        $html .= '<meta property="og:url" content="' . htmlspecialchars($this->currentUrl) . '">' . "\n";
        $html .= '<meta property="og:site_name" content="Yapı Dekorasyon">' . "\n";
        $html .= '<meta property="og:locale" content="tr_TR">' . "\n";
        $html .= '<meta property="og:image" content="' . htmlspecialchars($image) . '">' . "\n";
        
        return $html;
    }

    /**
     * Twitter Card etiketleri
     */
    private function generateTwitterCard(string $title, string $description): string
    {
         $image = $this->pageData['featured_image'] ?? '/assets/images/hero-bg.jpg';
         
          if (strpos($image, 'http') !== 0) {
            $image = rtrim($this->baseUrl, '/') . '/' . ltrim($image, '/');
        }

        $html = "\n<!-- Twitter Card -->\n";
        $html .= '<meta name="twitter:card" content="summary_large_image">' . "\n";
        $html .= '<meta name="twitter:title" content="' . htmlspecialchars($title) . '">' . "\n";
        $html .= '<meta name="twitter:description" content="' . htmlspecialchars($description) . '">' . "\n";
        $html .= '<meta name="twitter:image" content="' . htmlspecialchars($image) . '">' . "\n";
        
        return $html;
    }

    // =========================================================================
    // CANONICAL URL
    // =========================================================================

    /**
     * Canonical URL üret
     */
    public function generateCanonical(): string
    {
        $canonicalUrl = $this->currentUrl;
        
        return '<link rel="canonical" href="' . htmlspecialchars($canonicalUrl) . '">' . "\n";
    }

    // =========================================================================
    // JSON-LD SCHEMA
    // =========================================================================

    /**
     * Sayfa tipine göre JSON-LD schema üret
     */
    public function generateSchema(): string
    {
        $schemas = [];
        
        // Her sayfada Organization schema
        $schemas[] = $this->getOrganizationSchema();
        
        // Sayfa tipine göre ek schemalar
        switch ($this->pageType) {
            case 'home':
                $schemas[] = $this->getLocalBusinessSchema();
                $schemas[] = $this->getWebSiteSchema();
                break;
            
            case 'service':
                $schemas[] = $this->getProductSchema();
                $schemas[] = $this->getBreadcrumbSchema();
                break;
            
            case 'services':
                $schemas[] = $this->getItemListSchema();
                break;
            
            case 'contact':
                $schemas[] = $this->getContactPointSchema();
                break;
            
            case 'about':
                $schemas[] = $this->getAboutPageSchema();
                break;
            
            case 'city':
                $schemas[] = $this->getLocalBusinessSchema();
                $schemas[] = $this->getServiceAreaSchema();
                break;
                
            case 'blog-detail':
                 $schemas[] = $this->getArticleSchema();
                 $schemas[] = $this->getBreadcrumbSchema();
                 break;
        }
        
        $html = "\n<!-- JSON-LD Structured Data -->\n";
        foreach ($schemas as $schema) {
            $html .= '<script type="application/ld+json">' . "\n";
            $html .= json_encode($schema, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES | JSON_PRETTY_PRINT);
            $html .= "\n</script>\n";
        }
        
        return $html;
    }

    /**
     * Organization Schema
     */
    private function getOrganizationSchema(): array
    {
        return [
            '@context' => 'https://schema.org',
            '@type' => 'Organization',
            'name' => $this->businessInfo['name'],
            'legalName' => $this->businessInfo['legalName'],
            'url' => $this->baseUrl,
            'logo' => rtrim($this->baseUrl, '/') . $this->businessInfo['logo'],
            'foundingDate' => $this->businessInfo['foundingDate'],
            'contactPoint' => [
                '@type' => 'ContactPoint',
                'telephone' => $this->businessInfo['telephone'],
                'contactType' => 'customer service',
                'availableLanguage' => 'Turkish'
            ],
            'sameAs' => $this->businessInfo['sameAs']
        ];
    }

    /**
     * LocalBusiness Schema
     */
    private function getLocalBusinessSchema(): array
    {
        return [
            '@context' => 'https://schema.org',
            '@type' => 'HomeAndConstructionBusiness',
            '@id' => $this->baseUrl . '/#localbusiness',
            'name' => $this->businessInfo['name'],
            'description' => $this->businessInfo['description'],
            'url' => $this->baseUrl,
            'telephone' => $this->businessInfo['telephone'],
            'email' => $this->businessInfo['email'],
            'image' => rtrim($this->baseUrl, '/') . $this->businessInfo['image'],
            'logo' => rtrim($this->baseUrl, '/') . $this->businessInfo['logo'],
            'priceRange' => $this->businessInfo['priceRange'],
            'address' => [
                '@type' => 'PostalAddress',
                'streetAddress' => $this->businessInfo['address']['streetAddress'],
                'addressLocality' => $this->businessInfo['address']['addressLocality'],
                'addressRegion' => $this->businessInfo['address']['addressRegion'],
                'postalCode' => $this->businessInfo['address']['postalCode'],
                'addressCountry' => $this->businessInfo['address']['addressCountry']
            ],
            'geo' => [
                '@type' => 'GeoCoordinates',
                'latitude' => $this->businessInfo['geo']['latitude'],
                'longitude' => $this->businessInfo['geo']['longitude']
            ],
            'openingHoursSpecification' => [
                '@type' => 'OpeningHoursSpecification',
                'dayOfWeek' => ['Monday', 'Tuesday', 'Wednesday', 'Thursday', 'Friday', 'Saturday'],
                'opens' => '09:00',
                'closes' => '18:00'
            ],
            'areaServed' => array_map(function($city) {
                return [
                    '@type' => 'City',
                    'name' => $city
                ];
            }, $this->businessInfo['areaServed']),
            'hasOfferCatalog' => [
                '@type' => 'OfferCatalog',
                'name' => 'Tadilat ve Dekorasyon Hizmetleri',
                'itemListElement' => [
                    ['@type' => 'Offer', 'itemOffered' => ['@type' => 'Service', 'name' => 'Anahtar Teslim Tadilat']],
                    ['@type' => 'Offer', 'itemOffered' => ['@type' => 'Service', 'name' => 'Mutfak Yenileme']],
                    ['@type' => 'Offer', 'itemOffered' => ['@type' => 'Service', 'name' => 'Banyo Tasarımı']],
                    ['@type' => 'Offer', 'itemOffered' => ['@type' => 'Service', 'name' => 'İç Mimarlık']]
                ]
            ]
        ];
    }

    /**
     * WebSite Schema
     */
    private function getWebSiteSchema(): array
    {
        return [
            '@context' => 'https://schema.org',
            '@type' => 'WebSite',
            'name' => $this->businessInfo['name'],
            'url' => $this->baseUrl,
            'potentialAction' => [
                '@type' => 'SearchAction',
                'target' => $this->baseUrl . '/arama?q={search_term_string}',
                'query-input' => 'required name=search_term_string'
            ]
        ];
    }

    /**
     * Product/Service Schema
     */
    private function getProductSchema(): array
    {
        $name = $this->pageData['name'] ?? 'Tadilat Hizmeti';
        $description = $this->pageData['short_description'] ?? $this->pageData['description'] ?? "İzmir {$name} hizmetleri.";
        $image = $this->pageData['featured_image'] ?? '/assets/images/default-service.jpg';
        $slug = $this->pageData['slug'] ?? 'tadilat';
        
        if (strpos($image, 'http') !== 0) {
            $image = rtrim($this->baseUrl, '/') . '/' . ltrim($image, '/');
        }
        
        return [
            '@context' => 'https://schema.org',
            '@type' => 'Service',
            'name' => 'İzmir ' . $name,
            'description' => $description,
            'image' => $image,
            'provider' => [
                '@type' => 'HomeAndConstructionBusiness',
                'name' => 'Yapı Dekorasyon'
            ],
            'areaServed' => [
                '@type' => 'City',
                'name' => 'İzmir'
            ],
             'offers' => [
                '@type' => 'Offer',
                'url' => $this->baseUrl . '/hizmetlerimiz/' . $slug,
                'priceCurrency' => 'TRY',
                'availability' => 'https://schema.org/InStock',
                'seller' => [
                    '@type' => 'Organization',
                    'name' => 'Yapı Dekorasyon'
                ]
            ]
        ];
    }

    /**
     * Breadcrumb Schema
     */
    private function getBreadcrumbSchema(): array
    {
        $items = [
            ['name' => 'Ana Sayfa', 'url' => $this->baseUrl],
        ];
        
        if ($this->pageType == 'service') {
              $items[] = ['name' => 'Hizmetlerimiz', 'url' => $this->baseUrl . '/hizmetlerimiz'];
               if (isset($this->pageData['name'])) {
                $items[] = [
                    'name' => $this->pageData['name'],
                    'url' => $this->baseUrl . '/hizmetlerimiz/' . ($this->pageData['slug'] ?? '')
                ];
            }
        } elseif ($this->pageType == 'blog-detail') {
             $items[] = ['name' => 'Blog', 'url' => $this->baseUrl . '/blog'];
              if (isset($this->pageData['title'])) {
                $items[] = [
                    'name' => $this->pageData['title'],
                    'url' => $this->currentUrl
                ];
            }
        }
        
        return [
            '@context' => 'https://schema.org',
            '@type' => 'BreadcrumbList',
            'itemListElement' => array_map(function($item, $index) {
                return [
                    '@type' => 'ListItem',
                    'position' => $index + 1,
                    'name' => $item['name'],
                    'item' => $item['url']
                ];
            }, $items, array_keys($items))
        ];
    }

    /**
     * ItemList Schema (Hizmetler)
     */
    private function getItemListSchema(): array
    {
        try {
            $services = $this->db->fetchAll(
                "SELECT name, slug FROM services WHERE is_active = 1 ORDER BY display_order LIMIT 20"
            );
        } catch (Exception $e) {
            $services = [];
        }
        
        return [
            '@context' => 'https://schema.org',
            '@type' => 'ItemList',
            'name' => 'Tadilat ve Dekorasyon Hizmetleri',
            'numberOfItems' => count($services),
            'itemListElement' => array_map(function($service, $index) {
                return [
                    '@type' => 'ListItem',
                    'position' => $index + 1,
                    'name' => $service['name'],
                    'url' => $this->baseUrl . '/hizmetlerimiz/' . $service['slug']
                ];
            }, $services, array_keys($services))
        ];
    }

    /**
     * ContactPage Schema
     */
    private function getContactPointSchema(): array
    {
        return [
            '@context' => 'https://schema.org',
            '@type' => 'ContactPage',
            'name' => 'Yapı Dekorasyon İletişim',
            'url' => $this->baseUrl . '/iletisim',
            'mainEntity' => [
                '@type' => 'HomeAndConstructionBusiness',
                'name' => $this->businessInfo['name'],
                'telephone' => $this->businessInfo['telephone'],
                'email' => $this->businessInfo['email'],
                'address' => [
                    '@type' => 'PostalAddress',
                    'streetAddress' => $this->businessInfo['address']['streetAddress'],
                    'addressLocality' => $this->businessInfo['address']['addressLocality'],
                    'addressRegion' => $this->businessInfo['address']['addressRegion'],
                    'postalCode' => $this->businessInfo['address']['postalCode'],
                    'addressCountry' => $this->businessInfo['address']['addressCountry']
                ]
            ]
        ];
    }

    /**
     * AboutPage Schema
     */
    private function getAboutPageSchema(): array
    {
        return [
            '@context' => 'https://schema.org',
            '@type' => 'AboutPage',
            'name' => 'Yapı Dekorasyon Hakkında',
            'url' => $this->baseUrl . '/hakkimizda',
            'mainEntity' => [
                '@type' => 'Organization',
                'name' => $this->businessInfo['name'],
                'foundingDate' => $this->businessInfo['foundingDate'],
                'description' => $this->businessInfo['description']
            ]
        ];
    }

    /**
     * ServiceArea Schema
     */
    private function getServiceAreaSchema(): array
    {
        $cityName = $this->pageData['name'] ?? 'İzmir';
        
        return [
            '@context' => 'https://schema.org',
            '@type' => 'Service',
            'name' => $cityName . ' Tadilat ve Dekorasyon Hizmetleri',
            'provider' => [
                '@type' => 'HomeAndConstructionBusiness',
                'name' => $this->businessInfo['name']
            ],
            'areaServed' => [
                '@type' => 'City',
                'name' => $cityName
            ],
            'serviceType' => 'Ev Yenileme ve Tadilat'
        ];
    }
    
    /**
     * Article Schema (Blog)
     */
    private function getArticleSchema(): array {
        $title = $this->pageData['title'] ?? 'Blog Yazısı';
        $desc = $this->pageData['short_description'] ?? '';
        $image = $this->pageData['image'] ?? '/assets/images/default-blog.jpg';
         if (strpos($image, 'http') !== 0) {
            $image = rtrim($this->baseUrl, '/') . '/' . ltrim($image, '/');
        }
        $datePublished = $this->pageData['created_at'] ?? date('c');
        
        return [
            '@context' => 'https://schema.org',
            '@type' => 'Article',
            'headline' => $title,
            'description' => $desc,
            'image' => $image,
            'author' => [
                '@type' => 'Organization',
                'name' => 'Yapı Dekorasyon'
            ],
            'publisher' => [
                 '@type' => 'Organization',
                 'name' => 'Yapı Dekorasyon',
                 'logo' => [
                     '@type' => 'ImageObject',
                     'url' => rtrim($this->baseUrl, '/') . $this->businessInfo['logo']
                 ]
            ],
            'datePublished' => $datePublished,
             'mainEntityOfPage' => [
                '@type' => 'WebPage',
                '@id' => $this->currentUrl
            ]
        ];
    }

    // =========================================================================
    // TAM SEO ÇIKTISI
    // =========================================================================

    /**
     * Tüm SEO çıktısını tek seferde üret
     */
    public function render(): string
    {
        $output = '';
        $output .= $this->generateMeta();
        $output .= $this->generateCanonical();
        $output .= $this->generateSchema();
        return $output;
    }
}

/**
 * Helper fonksiyon
 */
function seo(): SeoManager
{
    static $instance = null;
    if ($instance === null) {
        $instance = new SeoManager();
    }
    return $instance;
}
